# R program for calculating the liquid phase and gas phase transfer velocities for any volatile compound given the relevant physical and chemical data

## this version submitted as supplementary material to Johnson and Liss, OSD, Dec 2009. 


## first of all read in the data file with each of the gases of interest
# note: if Vb is specified, then this value is used in diffusion coefficient calculations rather than the Schroeder method
# KH and tVar shoudl be taken from Rolf Sander's compilation of Henry's lay constants (http://www.mpch-mainz.mpg.de/~sander/res/henry.html) or other values presented in the same format

compounds<-read.table("compounds.dat", header = TRUE, row.names = 1)

#let's define a range of temperature and windspeed over which to calculate transfer velocities etc by default

t = c(-5,0,5,10,15,20,25,30,35)
WS = c(0.1,0.5,1,2,5,10,20,30)


##########################################################
#              MOLAR VOLUME AT BOILING POINT             #
##########################################################
 


Vb <- function(compound){
	## Calculate the molar volume at boiling point, Vb, using the Schroeder method, or just take the overrulling value from compounds.dat if specified
	# note: for compounds containing elements other than C,H,O,N,S, Cl, Br, F, and I, the schroeder method won't work so Vb must be specified
	# db, tb and rings should be the number of double and triple bonds, and ring features in the moelcule respectively. 
    	if (compounds[compound,"rings"]>0) ringval=-7 else ringval=0
    	if (compounds[compound,"Vb"]>0) compounds[compound,"Vb"] else 
        	7*(compounds[compound,"C"]+compounds[compound,"H"]+compounds[compound,"O"]+compounds[compound,"N"]+compounds[compound,"db"])+14*compounds[compound,"tb"]+31.5*compounds[compound,"Br"]+24.5*compounds[compound,"Cl"]+10.5*compounds[compound,"F"]+38.5*compounds[compound,"I"]+21*compounds[compound,"S"]+ringval 
}


##########################################################
#      LIQUID PHASE TRANSFER VELOCITY CALCULATIONS       #
##########################################################


n_0 <- function(T){
	#Calculate the viscosity of pure water in cP (mPa s) according to LaLiberte 2007 (doi:10.1021/je0604075). Requires T in celcius
    	(T+246)/(137.37+(5.2842*T)+(0.05594*(T^2)))
}



n_sw <- function(T,S,return_relative_viscotiy=0){
	#calculate the dynamic viscosity of seawater in cP using the viscosity model / mixing rule approach of LaLiberte 2007
	#to return relative viscosity (i.e. viscosity scaling factor) for comparison with other studies, pass n_sw a final argument of "1" after T and S e.g. n_sw(10,35,1), otherwise, for absolute viscosities, leave it out (e.g. n_sw(10,35))
    	#read in a data file containing the mass fraction of each constituent solute in seawater per salinity unit in per mil and the coefficients determined by LaLiberte 2007 for each salt
    	sw_cmf<-read.table("sw_constituent_mass_fractions.dat", header = TRUE, row.names = 1)
    	#w_i_ln_n_i_tot is the sum of the product of the mass fraction and the natural log of the viscosities contributed by each solute individually (see LaLiberte 2007, equation 8)
    	w_i_ln_n_i_tot <-0
    	#sum up the mass fractions to get water mass fraction
    	w_i_tot<-0
    	for (salt in row.names(sw_cmf)){
       		w_i <- sw_cmf[salt,"massfraction"]*S/1000
       		w_i_tot<-w_i_tot + w_i
    	}
    	for (salt in row.names(sw_cmf)){
       		w_i <- sw_cmf[salt,"massfraction"]*S/1000
       		v1 <- sw_cmf[salt,"v1"]
       		v2 <- sw_cmf[salt,"v2"]
       		v3 <- sw_cmf[salt,"v3"]
       		v4 <- sw_cmf[salt,"v4"]
       		v5 <- sw_cmf[salt,"v5"]
       		v6 <- sw_cmf[salt,"v6"]
       		#wi_tot is used here as eq 12 in LaLiberte requires (1-w_w), which is equivalent. Using this term seems initially counterintuitive - one might expect to use w_i here for each solute individually. However,  "at any given solute concentration, the solute viscosity increases with the total solute content" so 1-w_w (or w_i_tot) is the correct term - pers. comm. from Marc LaLiberte
       		n_i<-(exp(((v1*w_i_tot^v2)+v3)/((v4*T) + 1)))/((v5*(w_i_tot^v6))+1)
       		w_i_ln_n_i_tot <- w_i_ln_n_i_tot + (w_i*log(n_i))
    	}
    	ln_n_m<- (1-w_i_tot)*log(n_0(T))+w_i_ln_n_i_tot
    	if (return_relative_viscotiy==0) exp(ln_n_m) else
        	exp(ln_n_m)/n_0(T)
}

n_sw_hardy <- function(T,n_zero=1){
	#Non-salinity dependent (gives values at fixed S=35) Hardy (see ITTC 2006) method for calculating viscosity - not for use in calculations, purely for comparison with Laliberte method.
	# η = η0 * K/[1+0.03338.T + 0.00018325.T2]
	# use n_zero = 1 to use Laliberte n_0 or n_zero=0 for Hardy n_0
    	if (n_zero==0) n0<- 1.787 else
        	n0 <- n_0(T)
    	n0*1.052/(1+(0.03338*T)+(0.00018325*(T^2)))
}

p_sw <- function(T,S){
	## Calculate the density of seawater in kg/m3 according to millero and Poisson (1981)
    	#coefficients for millero calc
    	A <- 0.824493-(0.0040899*T)+(0.000076438*(T^2))-(0.00000082467*(T^3))+(0.0000000053875*(T^4))
    	B <- -0.00572466+(0.00010277*T)-(0.0000016546*(T^2))
    	C <- 0.00048314
    	# density of pure water
    	p_o <- 999.842594+(0.06793952*T)-(0.00909529*(T^2))+(0.0001001685*(T^3))-(0.000001120083*(T^4))+(0.000000006536332*(T^5))
    	# return salinity-and-temperature-dependent density of seawater (at atmospheric pressure)
    	p_o+(A*S)+(B*(S^(3/2)))+(C*S)
} 

v_sw <- function(T,S) {
	#calculate kinmatic viscosity of seawater in cm/s for Schmidt number calculation
    	# dynamic viscosity - convert to S.I. (Kg/m/s)
    	n = n_sw(T,S)/1000
    	# density already in S.I.
    	p = p_sw(T,S)
    	#multiply by 10000 to go from m2/s to cm2/s
    	10000*n/p
}

#################################################
#all diffusion coefficients calculated in cm^2/s#
#################################################

diff_HM <- function(compound,T,S,vs=1){
	#Hayduk and Minhas (1982) diffusion coefficient calculation
	#in this and subsequent calculations, there is the ability to apply a vicosity scaling factor (vs)
    	EpsilonStar <- (9.58/Vb(compound))-1.12
    	1.25e-8*(Vb(compound)^(-0.19)-0.292)*((T+273.15)^1.52)*((vs*n_sw(T,S))^EpsilonStar)
}

schmidt_HM <- function(compound,T,S,vs=1){
	#calculate schmidt number from HM diffusivity
	(vs*v_sw(T,S))/diff_HM(compound,T,S,vs)
}

diff_WC <- function(compound,T,S,vs=1){
	#Wilkie and Chang (1955) diffusion coefficient
    	# associaction factor of solvent (2.6 in the case of water according to Poling 2001; although Wanninkhof suggests 2.26)
    	phi <- 2.6
    	((T+273.15)*7.4e-8*(phi*18.01)^0.5)/((vs*n_sw(T,S))*(Vb(compound)^0.6))
}

schmidt_WC <- function(compound,T,S,vs=1){
	#calculate schmidt number from WC diffusivity
    	(vs*v_sw(T,S))/diff_WC(compound,T,S,vs)
}

schmidt <- function(compound,T,S,vs=1){
	#calculate mean schmidt number in water
    	mean_diff<-0.5*(diff_WC(compound,T,S,vs)+diff_HM(compound,T,S,vs))
        vs*v_sw(T,S)/mean_diff
}

schmidt_out<-function(compound,T,S,schmidt=0,vs=1){
	#output schmidt number T dependence for a list of T
	#calculated with mean schmidt number by default, or specify 1 for Hayduck and Minhas or 2 for Wilkie and Chang
    	if (schmidt==0) schmidt(compound,T,S,vs) else
    		if (schmidt==1) schmidt_HM(compound,T,S,vs) else
    			schmidt_WC(compound,T,S,vs)
}


kw <- function(compound,T,ws,S,normalize=0,schmidt_formulation=0,vs=1){
	# calculate kw according to Nightingale et al 2000
	# note k600 not k660 for their study
    	if (normalize!=0) schmidt_number<-normalize else
        	schmidt_number<-schmidt_out(compound,T,S,schmidt_formulation,vs)
    	(((0.222*ws^2)+0.333*ws)*(schmidt_number/600)^(-0.5))/(100*3600)
}


###############################
# other kw parameterisations  #
###############################


Wannkw <- function (compound,T,ws,S,normalize=0,schmidt_formulation=0,vs=1){
	#calculate kw transfer velocity in m/s according to Wanninkhof 1992
	#specifying a 'normalize' value allows the calculation of the transfer velocity for a user-specified value of schmidt number
    	if (normalize!=0) schmidt_number<-normalize else
    	schmidt_number<-schmidt_out(compound,T,S,schmidt_formulation,vs)
    	(0.31*ws^2*((schmidt_number/660)^(-0.5)))/(100*3600)
}

Lissmer_approx_kw <- function(compound,T,ws,S,normalize=0,schmidt_formulation=0,vs=1){
	#calculate kw in m/s according to liss and merlivat 
	# note k600 not k660
	# in order to make calculations easier, an exponential relationship has been fitted to the Liss and Merlivat Curve (this isn't ideal, but as L+M is only ever used as a bottom bound for a range of possible transfer velocity values, it does the job)
    	if (normalize!=0) schmidt_number<-normalize else
        	schmidt_number<-schmidt_out(compound,T,S,schmidt_formulation,vs)
    	(0.075*ws^2.25*((schmidt_number/600)^(-0.5))/(100*3600))
}

Sweeneykw <- function (compound,T,ws,S,normalize=0,schmidt_formulation=0,vs=1){
	#calculate kw according to Sweeney 2007
    	if (normalize!=0) schmidt_number<-normalize else
    		schmidt_number<-schmidt_out(compound,T,S,schmidt_formulation,vs)
    	(0.27*ws^2*((schmidt_number/660)^(-0.5)))/(100*3600)
}


########################################################
#              HENRY'S LAW COEFFICIENTS                #
########################################################

KH0 <- function(compound,T=25){
	#Calculate Henry's law constant at a given T in pure water according to Sander (1999)
    	12.2/((273.15+T)*(compounds[compound,"KH"])*exp((compounds[compound,"tVar"])*((1/(T+273.15))-(1/298.15))))
}


Ks<-function(compound){
	theta = (7.3353282561828962e-04 + (3.3961477466551352e-05*log(KH0(compound))) + (-2.4088830102075734E-06*(log(KH0(compound)))^2) + (1.5711393120941302E-07*(log(KH0(compound)))^3))
	theta*log(Vb(compound))
}

K_H_factor <-function(compound,S){
	#calculate salinity-dependent salting-out scaling factor for KH0 (see manuscript)
    	10^(Ks(compound)*S)
}

KH <- function(compound,T,S){
	#Calculate Henry's law constant at a given T and Salinity
    	KH0(compound,T)*K_H_factor(compound,S)
}

KH_Molar_per_atmosphere <- function(compound,T,S){
 # not used in scheme - just useful to have!
	# calculate the Henry's law constant in M/atm from Sander data
	# applying the salting out factor above
    	(compounds[compound,"KH"]*exp((compounds[compound,"tVar"])*((1/(T+273.15))-(1/298.15))))/K_H_factor(compound,S)
}


########################################################
#       GAS PHASE TRANSFER VELOCITY CALCULATIONS       #
########################################################


Fuller_D_air <- function(compound,T){
	#calculate diffusivity in air in cm2/sec
	#M_a is molar weight of air
	M_a <- 28.97
	M_b <- compounds[compound,"mw"]
	M_r <- (M_a + M_b)/(M_a*M_b)
	#assume 1ATM
	P <- 1
	#assume molar volume air is 20.1 cm3/mol
	V_a <- 20.1	
	(0.001*((T+273.15)^1.75)*sqrt(M_r))/(P*((V_a^(1/3))+(Vb(compound)^(1/3))))^2
}

D_air <- function(compound,T){
 #in case new diffusion coefficients are added in the future
	Fuller_D_air(compound,T)
} 

n_air <- function(T){
	# dynamic viscosity of saturated air according to Tsiligiris 2008
	SV_0 = 1.715747771e-5
	SV_1 = 4.722402075e-8
	SV_2 = -3.663027156e-10
	SV_3 = 1.873236686e-12
	SV_4 = -8.050218737e-14
	
	# in N.s/m^2 (Pa.s)
	u_m = SV_0+(SV_1*T)+(SV_2*T^2)+(SV_3*T^3)+(SV_4*T^4)
	u_m
}

p_air <- function(T){
	# density of saturated air according to Tsiligiris 2008 in kg/m^3
	SD_0 = 1.293393662
	SD_1 = -5.538444326e-3
	SD_2 = 3.860201577e-5
	SD_3 = -5.2536065e-7

 # in kg/m^3
	p = SD_0+(SD_1*T)+(SD_2*T^2)+(SD_3*T^3)
	p
}

v_air <- function(T) {
	#calculate kinmatic viscosity of air in cm2/s for Schmidt number calculation
    	# dynamic viscosity 
    	n = n_air(T)
    	# density 
    	p = p_air(T)
    	#multiply by 10000 to go from m2/s to cm2/s
    	10000*n/p
}

Sc_air <- function (compound,T){
	#calculate the schmidt number of a given gas in air
	v_air(T)/D_air(compound,T)
}



ka<-function(compound,ws,T){
 #ka in m/s
 u_star<-ws*sqrt(1e-4*(6.1+0.63*ws))
 ra_turb <-ws/(u_star)^2
	ra_diff <-(5/u_star)*((Sc_air(compound,T))^(2/3))
 1e-3 +1/(ra_turb + ra_diff)
}


#########################################
#   other parameterisations of ka       #
#########################################

Duce_ka <- function(compound,ws){
	#calculate ka gas phase transfer velocity according to Duce et al 1991
    	ws/(770+(45*((compounds[compound,"mw"])^(1/3))))
}

Liss_ka <- function(ws){
	#calculate ka according to Liss 1973 (non compound-specific)
	(0.005+(0.21*ws))/100
}

MackayYeun_ka <- function(compound,ws,T)
	#ka according to Mackay and Yeun 1983
	1e-3 + (46.2e-5*sqrt(6.1+(0.63*ws))*ws*(Sc_air(compound,T))^-0.67)

Shahin_ka <- function(compound,ws,T){
	#calculate transfer velocity from gas phase diffusivity in air according to Shahin et al 2002 in m/s
	#checked against shahin data - they find a ka of ~3 cm/s at 6m/s wind - approx 5 times that of Duce!)-
	(sqrt(D_air(compound,T))*((0.98*ws)+1.26))/100
}




#####################################################################
#               TOTAL TRANSFER VECLOCITIES                          #
#####################################################################



rl <- function(compound,T,ws,S,schmidt_formulation=0){
	#calculate rl (=1/kw) after Liss and Slater 1974
    	1/kw(compound,T,ws,S,0,schmidt_formulation)
}

rl_prime <- function(compound,T,ws,S,schmidt_formulation=0){
	#rl_prime = KH/kw
    	KH(compound,T,S)/kw(compound,T,ws,S,0,schmidt_formulation)
}

rg <- function(compound,T,ws,S){
	#calculate rg (=1/KH*ka) after Liss and Slater
    	1/(KH(compound,T,S)*ka(compound,ws,T))
}

rg_prime <- function(compound,ws,T){
	#rg_prime = 1/ka 
    	1/ka(compound,ws,T)
}


Kw <- function(compound,T,ws,S){
	#calculate total transfer velocity (Kw) after Liss and Slater
    	1/(rg(compound,T,ws,S)+rl(compound,T,ws,S))
}

Ka <- function(compound,T,ws,S){
	#calculate total transfer velocity (Ka) after Liss and Slater
    	1/(rg_prime(compound,ws,T)+rl_prime(compound,T,ws,S))
}


#THE END

