% linN model (multiple reaches, with friction refinement)
%------------------------------------------------------------
% Modified based on Toffolon and Savenije (2011) by Huayang Cai
% References:Toffolon, M., and Savenije, H. H. G.: 
%       Revisiting linearized one-dimensional tidal propagation,
%       J. Geophys.Res., 116, C07007, doi:10.1029/2010JC006616, 2011.
%------------------------------------------------------------
% Outputs:
% - eta: tidal amplitude [m]
% - upsilon: velocity amplitude [m/s]
% - mu: velocity number [-]
% - deltaA: damping rate for the tidal amplitude [-]
% - deltaV: damping rate for the velocity amplitude [-]
% - lambdaA: celerity number for the water level [-]
% - lambdaV: celerity number for the velocity [-]
% - PhiA:phase difference of water level relative to the estuary mouth [-]
%
% Inputs for the model:
% - Le: dimensional channel length [m]
% - eta0: tidal amplitude at the estuary mouth [m]
% - T: tidal period [s]
% - D0: depth at the estuary mouth [m]
% - rs: storage width ratio [-]
% - K: Manning-Strickler friction coefficient [m^{1/3}s^{-1}]
% - Lb: width convergence length [m]
% - Lz: depth convergence length [m]
% - dx: distance interval [m]
% - f: friction correction factor [-]
% - inf=1 infinite channel, otherwise finite channel
% Hypothesis:
% - uniform exponential convergence for both width and depth 
%   (they can be relaxed using variable gamma's)
function [eta,upsilon,PhiA,deltaA,lambdaA]=f_linN(Le,eta0,T,D0,rs,ks,Lb,Lz,dx,f,inf)
%% model parameters
g=9.81;         % gravity [m/s^2]
omega=2*pi/T;   % tidal frequency [s^-1]
kappa=f.*8./(3*pi); % Lorentz's constant [-]
kmax=100;       % max number of iterations [-]
toll=1e-6;       % tolerance to satisfy [-]
%% computation
N=Le/dx; %divide the whole estuary into N reaches
epsilon=eta0/D0; %dimensionless tidal amplitude at the estuary mouth
%solution of linear system
sys_sol=1; %=1 with matlab routine
%-------- multiple reaches solution ----------
xi=[0:N-1]*Le/N;
Ds=D0*exp(-xi/Lz);
Rs=-(rs-1)./Le.*xi+rs; % Rs interpolated linearly
Cs=sqrt(g*Ds./Rs); %Celerity
Ls=Cs/omega*2*pi; %length scale
alpha=Ds(2:N)./Ds(1:N-1);
Ch=ks/sqrt(g)*Ds.^(1/6);
L=diff([xi Le])./Ls;
chi=Rs.*epsilon.*Cs./(Ch.^2.*omega.*Ds);
gammaB=Ls./Lb./2./pi;
gammaZ=Ls./Lz./2./pi;
gamma=gammaB+gammaZ;
Gamma=1-(gamma/2).^2;
%-------- start iterations ----------
mu=chi*0+1;
for kk=1:kmax
    chi1=kappa.*mu.*chi;
    Delta=sqrt(-Gamma+1i*chi1);
    w1=gamma/2+Delta;
    w2=gamma/2-Delta;
    ew1=exp(w1.*L.*2.*pi);
    ew2=exp(w2.*L.*2.*pi); 
    M=zeros(2*N,2*N);P=zeros(2*N,1);
    for j=1:N-1
        je=2*j-1; %equation for water level
        M(je,2*j-1)=ew1(j);
        M(je,2*j)  =ew2(j);
        M(je,2*j+1)=alpha(j);
        M(je,2*j+2)=alpha(j);
        je=2*j;   %equation for velocity (U)
        M(je,2*j-1)=ew1(j)/w2(j);
        M(je,2*j)  =ew2(j)/w1(j);
        pp=alpha(j)^(1/2);
        M(je,2*j+1)=pp/w2(j+1);
        M(je,2*j+2)=pp/w1(j+1);
    end
    j=N;
    je=2*j-1; %equation for water level
    M(je,1)=1;
    M(je,2)=1;
    P(je)=1;
    je=2*j;   %equation for velocity
    M(je,2*N-1)=ew1(j)/w2(j);
    M(je,2*N)  =ew2(j)/w1(j);
    if(inf==1)
        M(je,2*N)=0;
    end
    
    if(sys_sol==1)
        %solution linear system
        aa=linsolve(M,P);
        a1=aa(1:2:2*N);
        a2=aa(2:2:2*N);
        a1=a1.'; %important a1'=transpose+conjugate,
        a2=a2.'; %          a1.'=only transpose
    else
        linsys; %explicit solution of the linear system
        a1=sa1;a2=sa2;
    end
    v1=1i*a1./w2; %important a1'=transpose+conjugate,
    v2=1i*a2./w1; %          a1.'=only transpose
    %velocity parameter
    V0=(v1+v2);
    mup=mu;
    mu=abs(V0);
    scarto=1-mu./mup;
    if(length(scarto)>1)
        scarto=std(scarto);
    else
        scarto=abs(scarto);
    end
    if(scarto<toll)
        kk2=kk; %number of iteration to satisfy tolerance
        break
    end
end
for j=1:N
    x=0:L(j)/10:L(j); 
    A=a1(j)*exp(2.*pi*w1(j)*x)+a2(j)*exp(2.*pi*w2(j)*x);
    H=epsilon*Ds(j)*abs(A);
    eta(j)=H(1);
    A_i=a2(j)*exp(2.*pi*w2(j)*x);
    H_i=epsilon*Ds(j)*abs(A_i);
    H_I(j)=H_i(1);
    A_r=a1(j)*exp(2.*pi*w1(j)*x);
    H_r=epsilon*Ds(j)*abs(A_r);
    H_R(j)=H_r(1);
    
    V=v1(j)*exp(2.*pi*w1(j)*x)+v2(j)*exp(2.*pi*w2(j)*x);
    U=epsilon*Cs(j)*abs(V);
    upsilon(j)=U(1);
    U_i=v2(j)*exp(2.*pi*w2(j)*x);
    V_i=epsilon*Cs(j)*abs(U_i);
    V_I(j)=V_i(1);
    U_r=v1(j)*exp(2.*pi*w1(j)*x);
    V_r=epsilon*Cs(j)*abs(U_r);
    V_R(j)=V_r(1);

    
    phiA=atan(imag(A)./real(A));
    phiV=atan(imag(V)./real(V));
    phi=phiV-phiA; 
    phi(phi<0)=phi(phi<0)+pi;
    if(j==N) 
        phi(end)=nan;
    end
    PhiA(j)=phiA(1);
    phi_linN(j)=phi(1);
    %%
    phiA_I=atan(imag(A_i)./real(A_i));
    phiV_I=atan(imag(U_i)./real(U_i));
    phi_I=phiV_I-phiA_I;
    phi_I(phi_I>pi/2)=phi_I(phi_I>pi/2)-pi;
    phi_I(phi_I<-pi/2)=phi_I(phi_I<-pi/2)+pi;
    if(j==N) 
        phi_I(end)=nan;
    end
    Phi_I(j)=phi_I(1);
    %%
    phiA_R=atan(imag(A_r)./real(A_r));
    phiV_R=atan(imag(U_r)./real(U_r));
    phi_R=phiV_R-phiA_R;
    phi_R(phi_R>pi/2)=phi_R(phi_R>pi/2)-pi;
    phi_R(phi_R<-pi/2)=phi_R(phi_R<-pi/2)+pi;
    if(j==N) 
        phi_R(end)=nan;
    end
    Phi_R(j)=phi_R(1);
    %***************************
    pp=(a1(j)*w1(j)+a2(j)*w2(j))/(a1(j)+a2(j));
deltaA(j)=real(pp);
lambdaA(j)=imag(pp);

pp=(v1(j)*w1(j)+v2(j)*w2(j))/(v1(j)+v2(j));
deltaV(j)=real(pp);
lambdaV(j)=imag(pp);

delta=(deltaA+deltaV)/2;
lambda=(lambdaA+lambdaV)/2;
end
end
