/***********************************************************
 *    Auxiliary functions used to implement thermodynamic  *
 *    quantities of ice Ih                                 *
 *                                                         *
 *    Details:                                             *
 *    Rainer Feistel & Wolfgang Wagner,                    *
 *    High-pressure thermodynamic Gibbs functions of ice   *
 *    and sea ice, Journal of Marine Research, 2005        *
 *    (in press)                                           *
 *                                                         *
 ***********************************************************/

#include "IceIh_FW_Aux.h"

namespace IceIh_FW_Aux {

const double g00=-632578.704355102,
             g01=0.655029997804786,
             g02=-1.89952376891314e-8,
             g03=3.40692612753936e-15,
             g04=-5.78593658679522e-22;

const dcomplex t1(3.71539090346389e-2, 5.10464771184122e-2),
               r1(45.951447199735, 65.223705014775),
               t2(0.345095829562823, 0.343315892017841),
               r20(-75.8695106343435, -80.9878506462645),
               r21(-5.75529765634353e-5, 5.09059011946526e-5),
               r22(2.39617513518116e-11, -2.73297877749166e-11);

const dcomplex compl_zero(0.0, 0.0);

const double sigma_absolute=189.13,
             sigma_IAPWS95=-3333.18160308627,
             Pt=611.657,    // Pa
             P0=101325.0,   // Pa
             Tt=273.16,     // K
             T0=273.15,     // K
             Tm=273.153,    // K
             TM=273.152518, // K
             p0=P0/Pt,      // dimensionless
             tau=273.16;    // dimensionless

/****************************************
  Auxiliary function.
  Units:
    P  - Pa
    g0 - J/kg
*****************************************/
double g0(double P)
{
  double p=P/Pt,x=p-p0;

  return g00+x*(g01+x*(g02+x*(g03+x*g04)));
}

/****************************************
  Auxiliary function.
  Units:
    P  - Pa
    dg0_dP - J/(kg*Pa)
*****************************************/
double dg0_dP(double P)
{
  double p=P/Pt,x=p-p0;

  return (g01+x*(2.0*g02+x*(3.0*g03+x*4.0*g04)))/Pt;
}

/****************************************
  Auxiliary function.
  Units:
    P  - Pa
    d2g0_dP2 - J/(kg*Pa*Pa)
*****************************************/
double d2g0_dP2(double P)
{
  double p=P/Pt,x=p-p0;

  return (2.0*g02+x*(6.0*g03+x*12.0*g04))/Pt/Pt;
}

/****************************************
  Auxiliary function.
  Units:
    P  - Pa
    d3g0_dP3 - J/(kg*Pa*Pa*Pa)
*****************************************/
double d3g0_dP3(double P)
{
  double p=P/Pt,x=p-p0;

  return (6.0*g03+24.0*x*g04)/Pt/Pt/Pt;
}

/****************************************
  Auxiliary function.
  Units:
    P  - Pa
    r2 - complex value
*****************************************/
dcomplex r2(double P)
{
  double p=P/Pt,x=p-p0;

  return r20+x*(r21+x*r22);
}

/****************************************
  Auxiliary function.
  Units:
    P  - Pa
    dr2_dP - complex value
*****************************************/
dcomplex dr2_dP(double P)
{
  double p=P/Pt,x=p-p0;

  return (r21+2.0*x*r22)/Pt;
}

/****************************************
  Auxiliary function.
  Units:
    P  - Pa
    d2r2_dP2 - complex value
*****************************************/
dcomplex d2r2_dP2(double P)
{
  return 2.0*r22/Pt/Pt;
}

/****************************************
  Auxiliary function.
  Units:
    P  - Pa
    d3r2_dP3 - complex value
*****************************************/
dcomplex d3r2_dP3(double P)
{
  return compl_zero;
}

} 
