/***********************************************************
 *    Gibbs thermodynamic potential, its derivatives and   *
 *    thermodynamic other properties of ice Ih             *
 *                                                         *
 *   Details:                                              *
 *    Rainer Feistel & Wolfgang Wagner,                    *
 *    High-pressure thermodynamic Gibbs functions of ice   *
 *    and sea ice, Journal of Marine Research, 2005        *
 *    (in press)                                           *
 *                                                         *
 ***********************************************************/

#include <iostream>
#include <fstream>
#include <complex>
using namespace std;
#include "IceIh_FW_Aux.h"

namespace IceIh_FW {

using namespace IceIh_FW_Aux;

inline double sqr(double x) { return x*x; }

/*******************************************
  Gibbs thermodynamic potential of ice Ih.
  Units:
    T - K
    P  - Pa
    ice_g - J/kg
********************************************/
double ice_g(double T, double P)
{
  double t=T/Tt;
  dcomplex w,z1,z2,a,b;

  a=t1-t;
  b=t1+t;
  z1=a*log(a)+b*log(b)-2.0*t1*log(t1)-t*t/t1;

  a=t2-t;
  b=t2+t;
  z2=a*log(a)+b*log(b)-2.0*t2*log(t2)-t*t/t2;

  w=r1*z1+r2(P)*z2;

  return g0(P)-sigma_IAPWS95*tau*t+tau*real(w);
}

/****************************************
  The derivative of g(T,P) with respect
  to P.
  Units:
    T - K
    P - Pa
    ice_dgdp - m**3/kg
*****************************************/
double ice_dgdp(double T, double P)
{
  double t=T/Tt;
  dcomplex w,z2,a,b;

  a=t2-t;
  b=t2+t;
  z2=a*log(a)+b*log(b)-2.0*t2*log(t2)-t*t/t2;

  w=dr2_dP(P)*z2;

  return dg0_dP(P)+tau*real(w);
}

/****************************************
  The second derivative of g(T,P) with
  respect to P.
  Units:
    T - K
    P - Pa
    ice_d2gdp2 - m**3/(kg*Pa)
*****************************************/
double ice_d2gdp2(double T, double P)
{
  double t=T/Tt;
  dcomplex w,z2,a,b;

  a=t2-t;
  b=t2+t;
  z2=a*log(a)+b*log(b)-2.0*t2*log(t2)-t*t/t2;

  w=d2r2_dP2(P)*z2;

  return d2g0_dP2(P)+tau*real(w);
}

/****************************************
  The third derivative of g(T,P) with
  respect to P.
  Units:
    T - K
    P - Pa
    ice_d3gdp3 - m**3/(kg*Pa*Pa)
*****************************************/
double ice_d3gdp3(double T, double P)
{
  return d3g0_dP3(P);
}

/****************************************
  The derivative of g(T,P) with
  respect to T.
  Units:
    T - K
    P - Pa
    ice_dgdt - J/(kg*K)
*****************************************/
double ice_dgdt(double T, double P)
{
  double t=T/Tt;
  dcomplex w,u1,u2,a,b;

  a=t1-t;
  b=t1+t;
  u1=-log(a)+log(b)-2.0*t/t1;

  a=t2-t;
  b=t2+t;
  u2=-log(a)+log(b)-2.0*t/t2;

  w=r1*u1+r2(P)*u2;

  return tau*(-sigma_IAPWS95+real(w))/Tt;
}

/****************************************
  The third derivative of g(T,P) with
  respect to T and twice P.
  Units:
    T - K
    P - Pa
    ice_d3gdtdp2 - m**3/(kg*K*Pa)
*****************************************/
double ice_d3gdtdp2(double T, double P)
{
  double t=T/Tt;
  dcomplex w,u2,a,b;

  a=t2-t;
  b=t2+t;
  u2=-log(a)+log(b)-2.0*t/t2;

  w=d2r2_dP2(P)*u2;

  return tau*real(w)/Tt;
}

/****************************************
  The second derivative of g(T,P) with
  respect to T.
  Units:
    T - K
    P - Pa
    ice_d2gdt2 - J/(kg*K*K)
*****************************************/
double ice_d2gdt2(double T, double P)
{
  double t=T/Tt;
  dcomplex w,u1,u2,a,b;

  a=t1-t;
  b=t1+t;
  u1=1.0/a+1.0/b-2.0/t1;

  a=t2-t;
  b=t2+t;
  u2=1.0/a+1.0/b-2.0/t2;

  w=r1*u1+r2(P)*u2;

  return tau*real(w)/Tt/Tt;
}

/****************************************
  The third derivative of g(T,P) with
  respect to T (twice) and P.
  Units:
    T - K
    P - Pa
    ice_d3gdt2dp - m**3/(kg*K**2)
*****************************************/
double ice_d3gdt2dp(double T, double P)
{
  double t=T/Tt;
  dcomplex w,u2,a,b;

  a=t2-t;
  b=t2+t;
  u2=1.0/a+1.0/b-2.0/t2;

  w=dr2_dP(P)*u2;

  return tau*real(w)/Tt/Tt;
}

/****************************************
  The second mixed derivative of g(T,P)
  with respect to T and P.
  Units:
    T - K
    P - Pa
    ice_d2gdtdp - m**3/(kg*K)
*****************************************/
double ice_d2gdtdp(double T, double P)
{
  double t=T/Tt;
  dcomplex w,u1,u2,a,b;

  a=t2-t;
  b=t2+t;
  u2=-log(a)+log(b)-2.0*t/t2;

  w=dr2_dP(P)*u2;

  return tau*real(w)/Tt;
}

/****************************************
  Specific volume of ice Ih.
  Units:
    T - K
    P - Pa
    ice_specific_volume - m**3/kg
*****************************************/
double ice_specific_volume(double T, double P)
{
  return ice_dgdp(T,P);
}

/****************************************
  Density of ice Ih.
  Units:
    T - K
    P - Pa
    ice_density - kg/m**3
*****************************************/
double ice_density(double T, double P)
{
  return 1.0/ice_dgdp(T,P);
}

/****************************************
  Cubic thermal expansion coeffficient
  of ice Ih.
  Units:
    T - K
    P - Pa
    ice_thermal_expansion - 1/K
*****************************************/
double ice_thermal_expansion(double T, double P)
{
  return ice_d2gdtdp(T,P)/ice_dgdp(T,P);
}

/****************************************
  Isothermal compressibility of ice Ih.
  Units:
    T - K
    P - Pa
    ice_isothermal_compressibility - 1/Pa
*****************************************/
double ice_isothermal_compressibility(double T, double P)
{
  return -ice_d2gdp2(T,P)/ice_dgdp(T,P);
}

/****************************************
  Adiabatic compressibility of ice Ih.
  Units:
    T - K
    P - Pa
    ice_isentropic_compressibility - 1/Pa
*****************************************/
double ice_isentropic_compressibility(double T, double P)
{
  double a,b,c,d;

  a=ice_d2gdtdp(T,P);
  b=ice_d2gdt2(T,P);
  c=ice_d2gdp2(T,P);
  d=ice_dgdp(T,P);

  return (a*a-b*c)/(d*b);
}

/****************************************
  The derivative of the adiabatic
  compressibility of ice Ih with respect
  to pressure P.
  Units:
    T - K
    P - Pa
    ice_disentropic_compressibilitydP - 1/Pa/Pa
*****************************************/
double ice_disentropic_compressibilitydP(double T, double P)
{
  double gP,gTP,gTT,gPP,gPPP,gPPT,gTTP;

  gP=ice_dgdp(T,P);

  gTP=ice_d2gdtdp(T,P);
  gTT=ice_d2gdt2(T,P);
  gPP=ice_d2gdp2(T,P);

  gPPP=ice_d3gdp3(T,P);
  gPPT=ice_d3gdtdp2(T,P);
  gTTP=ice_d3gdt2dp(T,P);

  return ((2.0*gTP*gPPT-gTTP*gPP-gTT*gPPP)*(gP*gTT)-
         (sqr(gTP)-gTT*gPP)*(gPP*gTT+gP*gTTP))/sqr(gP*gTT);
}

/********************************************
  Isochoric pressure coefficient of ice Ih.
  Units:
    T - K
    P - Pa
    ice_pressure_coefficient - 1/K
*********************************************/
double ice_pressure_coefficient(double T, double P)
{
  return -ice_d2gdtdp(T,P)/ice_d2gdp2(T,P)/P;
}

/****************************************
  Specific heat capacity at constant
  pressure of ice Ih.
  Units:
    T - K
    P - Pa
    ice_heat_capacity - J/(kg*K)
*****************************************/
double ice_heat_capacity(double T, double P)
{
  return -T*ice_d2gdt2(T,P);
}

/****************************************
  Specific free energy of ice Ih.
  Units:
    T - K
    P - Pa
    ice_free_energy - J/kg
*****************************************/
double ice_free_energy(double T, double P)
{
  return ice_g(T,P)-P*ice_dgdp(T,P);
}

/****************************************
  Specific entropy of ice Ih.
  Units:
    T - K
    P - Pa
    ice_entropy - J/(kg*K)
*****************************************/
double ice_entropy(double T, double P)
{
  return -ice_dgdt(T,P);
}

/****************************************
  Specific enthalpy of ice Ih.
  Units:
    T - K
    P - Pa
    ice_enthalpy - J/kg
*****************************************/
double ice_enthalpy(double T, double P)
{
  return ice_g(T,P)-T*ice_dgdt(T,P);
}

/****************************************
  Specific energy of ice Ih.
  Units:
    T - K
    P - Pa
    ice_internal_energy - J/kg
*****************************************/
double ice_internal_energy(double T, double P)
{
  return ice_g(T,P)-T*ice_dgdt(T,P)-P*ice_dgdp(T,P);
}

}

