Attribute VB_Name = "Gibbs04_mdl"
Option Explicit
'DefLng I-N: DefDbl A-H, O-Z

'Triple point:
Private Const Tt# = 273.16
Private Const Pt# = 611.657

Private Const P0# = 101325# '1 atm in Pa

Public Const ice_error_return = -9.99999999E+98

'auxiliary constants tau, psi
Private Const tau# = 273.16
Private Const psi# = 611.657

'Gibbs function coefficients
Private gcoeff(4) As Double
Private scoeff(0) As Double
Private rcoeff(2, 2) As CplxType
Private tcoeff(2, 0) As CplxType


Function ice_d3gdt3#(ByVal T#, ByVal P#)
On Error GoTo Errortrap

Dim CPC As CplxType

Dim Tc As CplxType
Dim r1 As CplxType
Dim r2 As CplxType
Dim t1 As CplxType
Dim t2 As CplxType

Dim y#, z#, dz#

y# = T# / Tt#  'T in K
Tc = Cplx_Num(y)

z# = P# / Pt#  'P in Pa
dz# = (P# - P0#) / Pt#

r1 = ddz_rk(1, dz#, 0)
t1 = ddz_tk(1, dz#, 0)
r2 = ddz_rk(2, dz#, 0)
t2 = ddz_tk(2, dz#, 0)

CPC = Cplx_Div(r1, Cplx_Power(Cplx_Diff(Tc, t1), 2))
CPC = Cplx_Diff(CPC, Cplx_Div(r1, Cplx_Power(Cplx_Sum(Tc, t1), 2)))
CPC = Cplx_Sum(CPC, Cplx_Div(r2, Cplx_Power(Cplx_Diff(Tc, t2), 2)))
CPC = Cplx_Diff(CPC, Cplx_Div(r2, Cplx_Power(Cplx_Sum(Tc, t2), 2)))

ice_d3gdt3# = CPC.Re * tau / Tt ^ 3

Exit Function

Errortrap:
ice_d3gdt3# = ice_error_return
End Function


Function ice_disentropic_compressibilitydp#(ByVal T#, ByVal P#)

' isothermal pressure derivative of adiabatic compressibility
' T in K
' P in Pa
' ice_disentropic_compressibilitydp in 1/Pa^2

Dim gp#, gtp#, gtt#, gpp#, gppp#, gtpp#, gttp#

gp# = ice_dgdp(T, P)
gtp# = ice_d2gdtdp(T, P)
gtt# = ice_d2gdt2(T, P)
gpp# = ice_d2gdp2(T, P)
gppp# = ice_d3gdp3(T, P)
gtpp# = ice_d3gdtdp2(T, P)
gttp# = ice_d3gdt2dp(T, P)

ice_disentropic_compressibilitydp = -(gtp ^ 2 / gtt - gpp) * gpp / gp ^ 2 + _
                 (2 * gtp * gtpp / gtt - gtp ^ 2 * gttp / gtt ^ 2 - gppp) / gp

End Function

Function ice_isentropic_compressibility#(ByVal T#, ByVal P#)
'all units in SI: J, kg, m, s, Pa, N etc
'T in K
'P in Pa
'adiabatic compressibility kappa in 1/Pa
'kappa(T,P) = -(1/v)*(dv/dP)|S = -(1/v)*(dv/dP)|T - T*(dv/dT)^2/(v * cp)

Dim gp#, gtt#, gtp#, gpp#

gp# = ice_g_FW04#(0, 1, T#, P#)
gtt# = ice_g_FW04#(2, 0, T#, P#)
gtp# = ice_g_FW04#(1, 1, T#, P#)
gpp# = ice_g_FW04#(0, 2, T#, P#)

ice_isentropic_compressibility# = (gtp ^ 2 - gtt * gpp) / (gp * gtt)
End Function


Function ice_isothermal_compressibility#(ByVal T#, ByVal P#)
'all units in SI: J, kg, m, s, Pa, N etc
'T in K
'P in Pa
'isothermal compressibility K in 1/Pa
'K(T,P) = -(1/v)*(dv/dP)

Dim gp#, gpp#

gp# = ice_g_FW04#(0, 1, T#, P#)
gpp# = ice_g_FW04#(0, 2, T#, P#)

ice_isothermal_compressibility# = -gpp / gp

End Function



Function ice_pressure_coefficient#(ByVal T#, ByVal P#)
'all units in SI: J, kg, m, s, Pa, N etc
'T in K
'P in Pa
'isochoric pressure coefficient beta in 1/K
'beta(T,P) =(1/P)*(dP/dT)|v

Dim gtp#, gpp#

gtp# = ice_g_FW04#(1, 1, T#, P#)
gpp# = ice_g_FW04#(0, 2, T#, P#)

ice_pressure_coefficient# = -gtp / (P * gpp)

End Function

Private Function ddz_cpoly(coeff() As CplxType, ByVal k%, ByVal z#, Optional ByVal deriv% = 0) As CplxType

'z-derivative (d/dz)^deriv of complex polynomial, poly = sum{ coeff(k, j)*z^j }

Dim polysum As CplxType
Dim cz As CplxType, zj As CplxType, cc As CplxType
Dim j%, l%

ddz_cpoly = Cplx_Num(0)
polysum = Cplx_Num(0)
cz = Cplx_Num(z)
zj = Cplx_Num(1)

For j = deriv% To UBound(coeff, 2)
  cc = Cplx_Mult(coeff(k, j), zj)
  For l = 1 To deriv%
    cc = Cplx_Factor(cc, j - l + 1)
  Next l
  polysum = Cplx_Sum(polysum, cc)
  If z = 0 Then Exit For
  If j < UBound(coeff, 2) Then zj = Cplx_Mult(zj, cz)
Next j

ddz_cpoly = polysum
End Function

Private Function ddz_poly(coeff() As Double, ByVal z#, Optional ByVal deriv% = 0) As Double

'z-derivative (d/dz)^deriv of real polynomial, poly = sum{ coeff(j)*z^j }

Dim polysum As Double
Dim cz As Double, zj As Double, cc As Double
Dim j%, l%

ddz_poly = 0
polysum = 0
cz = z
zj = 1#

For j = deriv% To UBound(coeff)
  cc = coeff(j) * zj
  For l = 1 To deriv%
    cc = cc * CDbl(j - l + 1)
  Next l
  polysum = polysum + cc
  If z = 0 Then Exit For
  If j < UBound(coeff) Then zj = zj * cz
Next j

ddz_poly = polysum
End Function


Function ice_g_FW04#(ByVal itemp%, ByVal ipres%, ByVal T#, ByVal P#)
'all units in SI: J, kg, m, s, Pa, N etc

Select Case ipres
  Case 0:

    Select Case itemp
      Case 0: ice_g_FW04 = ice_g#(T, P)
      Case 1: ice_g_FW04 = ice_dgdt(T, P)
      Case 2: ice_g_FW04 = ice_d2gdt2(T, P)
      Case Else: MsgBox "Derivative not supported"
    End Select

  Case 1:
    Select Case itemp
      Case 0: ice_g_FW04 = ice_dgdp(T, P)
      Case 1: ice_g_FW04 = ice_d2gdtdp(T, P)
      Case 2: ice_g_FW04 = ice_d3gdt2dp(T, P)
      Case Else: MsgBox "Derivative not supported"
    End Select

  Case 2:
    Select Case itemp
      Case 0: ice_g_FW04 = ice_d2gdp2(T, P)
      Case 1: ice_g_FW04 = ice_d3gdtdp2(T, P)
      Case Else: MsgBox "Derivative not supported"
    End Select

  Case 3: ice_g_FW04 = ice_d3gdp3(T, P)
  
  Case Else: MsgBox "Derivative not supported"

End Select

End Function


Function ice_g#(ByVal T#, ByVal P#)
On Error GoTo Errortrap

Dim Tc As CplxType
Dim r1 As CplxType
Dim t1 As CplxType
Dim r2 As CplxType
Dim t2 As CplxType
Dim tk As CplxType
Dim gc As CplxType

Dim term_tlnt1 As CplxType
Dim term_tlnt2 As CplxType

Dim y#, z#, dz#, g#

y# = T# / Tt#  'T in K
Tc = Cplx_Num(y)

z# = P# / Pt#  'P in Pa
dz# = (P# - P0#) / Pt#

r1 = ddz_rk(1, dz#, 0)
t1 = ddz_tk(1, dz#, 0)
r2 = ddz_rk(2, dz#, 0)
t2 = ddz_tk(2, dz#, 0)

tk = Cplx_Diff(t1, Tc)
gc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t1)
gc = Cplx_Sum(gc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t1, Cplx_Log(t1)), 2)
gc = Cplx_Diff(gc, tk)
term_tlnt1 = Cplx_Diff(gc, Cplx_Div(Cplx_Num(y ^ 2), t1))

tk = Cplx_Diff(t2, Tc)
gc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t2)
gc = Cplx_Sum(gc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t2, Cplx_Log(t2)), 2)
gc = Cplx_Diff(gc, tk)
term_tlnt2 = Cplx_Diff(gc, Cplx_Div(Cplx_Num(y ^ 2), t2))

gc = Cplx_Mult(r1, term_tlnt1)
g# = tau * gc.Re
gc = Cplx_Mult(r2, term_tlnt2)
g# = g# + tau * gc.Re
g# = g# - scoeff(0) * tau * y

ice_g# = g + ddz_poly(gcoeff(), dz)

Exit Function

Errortrap:
ice_g# = ice_error_return
End Function

Private Function ddz_rk(ByVal k%, ByVal z#, ByVal deriv%) As CplxType

'pressure derivative (d/dp)^deriv of r(k)
'z = (P-P0)/Pt
ddz_rk = ddz_cpoly(rcoeff(), k%, z#, deriv%)

End Function
Private Function ddz_tk(ByVal k%, ByVal z#, ByVal deriv%) As CplxType

'pressure derivative (d/dp)^deriv of t(k)
'z = (P-P0)/Pt
ddz_tk = ddz_cpoly(tcoeff(), k%, z#, deriv%)

End Function


Function ice_dgdt#(ByVal T#, ByVal P#)
On Error GoTo Errortrap

Dim dsdr1 As CplxType
Dim dsdr2 As CplxType
Dim dsdt1 As CplxType
Dim dsdt2 As CplxType
Dim sc As CplxType
Dim Tc As CplxType
Dim r1 As CplxType
Dim r2 As CplxType
Dim t1 As CplxType
Dim t2 As CplxType

Dim y#, z#, dz#

y# = T# / Tt#  'T in K
Tc = Cplx_Num(y)

z# = P# / Pt#  'P in Pa
dz# = (P# - P0#) / Pt#

r1 = ddz_rk(1, dz#, 0)
t1 = ddz_tk(1, dz#, 0)
r2 = ddz_rk(2, dz#, 0)
t2 = ddz_tk(2, dz#, 0)

dsdr1 = Cplx_Diff(Cplx_Log(Cplx_Diff(t1, Tc)), Cplx_Log(Cplx_Sum(t1, Tc)))
dsdr1 = Cplx_Sum(dsdr1, Cplx_Div(Cplx_Num(2 * y), t1))
dsdr2 = Cplx_Diff(Cplx_Log(Cplx_Diff(t2, Tc)), Cplx_Log(Cplx_Sum(t2, Tc)))
dsdr2 = Cplx_Sum(dsdr2, Cplx_Div(Cplx_Num(2 * y), t2))
sc = Cplx_Mult(r1, dsdr1)
sc = Cplx_Sum(sc, Cplx_Mult(r2, dsdr2))
ice_dgdt# = -(scoeff(0) + sc.Re) * tau / Tt

Exit Function

Errortrap:
ice_dgdt# = ice_error_return
End Function



Function ice_d3gdp3#(ByVal T#, ByVal P#)
On Error GoTo Errortrap

Dim d3r2dp3 As CplxType
Dim Tc As CplxType
Dim r2 As CplxType
Dim t2 As CplxType
Dim tk As CplxType
Dim vc As CplxType
Dim term_tlnt2 As CplxType

Dim y#, z#, dz#, d3gdz3#

y# = T# / Tt#  'T in K
Tc = Cplx_Num(y)

z# = P# / Pt#  'P in Pa
dz# = (P# - P0#) / Pt#

r2 = ddz_rk(2, dz#, 0)
t2 = ddz_tk(2, dz#, 0)

d3r2dp3 = Cplx_Factor(ddz_rk(2, dz#, 3), 1 / psi ^ 3)

tk = Cplx_Diff(t2, Tc)
vc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t2)
vc = Cplx_Sum(vc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t2, Cplx_Log(t2)), 2)
vc = Cplx_Diff(vc, tk)
term_tlnt2 = Cplx_Diff(vc, Cplx_Div(Cplx_Num(y ^ 2), t2))

d3gdz3# = ddz_poly(gcoeff(), dz, 3) / (psi ^ 3 * tau)

vc = Cplx_Mult(d3r2dp3, term_tlnt2)
d3gdz3# = d3gdz3# + vc.Re

ice_d3gdp3# = tau * d3gdz3#

Exit Function

Errortrap:
ice_d3gdp3# = ice_error_return
End Function


Function ice_d2gdt2#(ByVal T#, ByVal P#)
On Error GoTo Errortrap

Dim CPC As CplxType
Dim CP0 As CplxType
Dim Tc As CplxType
Dim r1 As CplxType
Dim r2 As CplxType
Dim t1 As CplxType
Dim t2 As CplxType

Dim y#, z#, dz#

y# = T# / Tt#  'T in K
Tc = Cplx_Num(y)

z# = P# / Pt#  'P in Pa
dz# = (P# - P0#) / Pt#

r1 = ddz_rk(1, dz#, 0)
t1 = ddz_tk(1, dz#, 0)
r2 = ddz_rk(2, dz#, 0)
t2 = ddz_tk(2, dz#, 0)

CPC = Cplx_Div(r1, Cplx_Diff(Tc, t1))
CPC = Cplx_Diff(CPC, Cplx_Div(r1, Cplx_Sum(Tc, t1)))
CPC = Cplx_Sum(CPC, Cplx_Div(r2, Cplx_Diff(Tc, t2)))
CPC = Cplx_Diff(CPC, Cplx_Div(r2, Cplx_Sum(Tc, t2)))
CP0 = Cplx_Sum(Cplx_Div(r1, t1), Cplx_Div(r2, t2))

ice_d2gdt2# = -(2# * CP0.Re + CPC.Re) * tau / Tt ^ 2

Exit Function

Errortrap:
ice_d2gdt2# = ice_error_return
End Function

Function ice_dgdp#(ByVal T#, ByVal P#)
On Error GoTo Errortrap

Dim dr1dp As CplxType
Dim dr2dp As CplxType
Dim dt1dp As CplxType
Dim dt2dp As CplxType
Dim Tc As CplxType
Dim r1 As CplxType
Dim r2 As CplxType
Dim t1 As CplxType
Dim t2 As CplxType
Dim tk As CplxType
Dim vc As CplxType

Dim term_lnt1 As CplxType
Dim term_lnt2 As CplxType
Dim term_tlnt1 As CplxType
Dim term_tlnt2 As CplxType

Dim y#, z#, dz#, dgdz#

y# = T# / Tt#  'T in K
Tc = Cplx_Num(y)

z# = P# / Pt#  'P in Pa
dz# = (P# - P0#) / Pt#

r1 = ddz_rk(1, dz#, 0)
t1 = ddz_tk(1, dz#, 0)
r2 = ddz_rk(2, dz#, 0)
t2 = ddz_tk(2, dz#, 0)

dr1dp = Cplx_Factor(ddz_rk(1, dz#, 1), 1 / psi) 'dr1/dP
dr2dp = Cplx_Factor(ddz_rk(2, dz#, 1), 1 / psi) 'dr2/dP
dt1dp = Cplx_Factor(ddz_tk(1, dz#, 1), 1 / psi) 'dt1/dP
dt2dp = Cplx_Factor(ddz_tk(2, dz#, 1), 1 / psi) 'dt2/dP

tk = Cplx_Diff(t1, Tc)
vc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t1)
vc = Cplx_Sum(vc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t1, Cplx_Log(t1)), 2)
vc = Cplx_Diff(vc, tk)
term_tlnt1 = Cplx_Diff(vc, Cplx_Div(Cplx_Num(y ^ 2), t1))

tk = Cplx_Diff(t2, Tc)
vc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t2)
vc = Cplx_Sum(vc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t2, Cplx_Log(t2)), 2)
vc = Cplx_Diff(vc, tk)
term_tlnt2 = Cplx_Diff(vc, Cplx_Div(Cplx_Num(y ^ 2), t2))

vc = Cplx_Sum(Cplx_Log(Cplx_Diff(t1, Tc)), Cplx_Log(Cplx_Sum(Tc, t1)))
vc = Cplx_Diff(vc, Cplx_Factor(Cplx_Log(t1), 2))
term_lnt1 = Cplx_Sum(vc, Cplx_Power(Cplx_Div(Tc, t1), 2))

vc = Cplx_Sum(Cplx_Log(Cplx_Diff(t2, Tc)), Cplx_Log(Cplx_Sum(Tc, t2)))
vc = Cplx_Diff(vc, Cplx_Factor(Cplx_Log(t2), 2))
term_lnt2 = Cplx_Sum(vc, Cplx_Power(Cplx_Div(Tc, t2), 2))

dgdz# = ddz_poly(gcoeff(), dz, 1) / (psi * tau)  'dg0/dp
vc = Cplx_Mult(dr1dp, term_tlnt1)
dgdz# = dgdz# + vc.Re

vc = Cplx_Mult(dr2dp, term_tlnt2)
dgdz# = dgdz# + vc.Re

vc = Cplx_Mult(dt1dp, term_lnt1)
vc = Cplx_Mult(r1, vc)
dgdz# = dgdz# + vc.Re

vc = Cplx_Mult(dt2dp, term_lnt2)
vc = Cplx_Mult(r2, vc)
dgdz# = dgdz# + vc.Re

ice_dgdp# = tau * dgdz#

Exit Function

Errortrap:
ice_dgdp# = ice_error_return
End Function

Function ice_d2gdtdp#(ByVal T#, ByVal P#)
On Error GoTo Errortrap

Dim dvdt As CplxType

Dim dr1dp As CplxType
Dim dr2dp As CplxType
Dim dt1dp As CplxType
Dim dt2dp As CplxType
Dim Tc As CplxType
Dim r1 As CplxType
Dim r2 As CplxType
Dim t1 As CplxType
Dim t2 As CplxType
Dim term_lnt1 As CplxType
Dim term_lnt2 As CplxType
Dim term_t1 As CplxType
Dim term_t2 As CplxType
Dim vc As CplxType

Dim y#, z#, dz#

y# = T# / Tt#  'T in K
Tc = Cplx_Num(y)

z# = P# / Pt#  'P in Pa
dz# = (P# - P0#) / Pt#

r1 = ddz_rk(1, dz#, 0)
t1 = ddz_tk(1, dz#, 0)
r2 = ddz_rk(2, dz#, 0)
t2 = ddz_tk(2, dz#, 0)

dr1dp = Cplx_Factor(ddz_rk(1, dz#, 1), 1 / psi) 'dr1/dP
dr2dp = Cplx_Factor(ddz_rk(2, dz#, 1), 1 / psi) 'dr2/dP
dt1dp = Cplx_Factor(ddz_tk(1, dz#, 1), 1 / psi) 'dt1/dP
dt2dp = Cplx_Factor(ddz_tk(2, dz#, 1), 1 / psi) 'dt2/dP

term_lnt1 = Cplx_Diff(Cplx_Log(Cplx_Sum(t1, Tc)), Cplx_Log(Cplx_Diff(t1, Tc)))
term_lnt1 = Cplx_Diff(term_lnt1, Cplx_Factor(Cplx_Div(Tc, t1), 2))

term_lnt2 = Cplx_Diff(Cplx_Log(Cplx_Sum(t2, Tc)), Cplx_Log(Cplx_Diff(t2, Tc)))
term_lnt2 = Cplx_Diff(term_lnt2, Cplx_Factor(Cplx_Div(Tc, t2), 2))

term_t1 = Cplx_Inv(Cplx_Diff(Tc, t1))
term_t1 = Cplx_Sum(term_t1, Cplx_Inv(Cplx_Sum(Tc, t1)))
term_t1 = Cplx_Sum(term_t1, Cplx_Div(Cplx_Num(2 * y), Cplx_Power(t1, 2)))

term_t2 = Cplx_Inv(Cplx_Diff(Tc, t2))
term_t2 = Cplx_Sum(term_t2, Cplx_Inv(Cplx_Sum(Tc, t2)))
term_t2 = Cplx_Sum(term_t2, Cplx_Div(Cplx_Num(2 * y), Cplx_Power(t2, 2)))

vc = Cplx_Mult(dr1dp, term_lnt1)
dvdt = vc
vc = Cplx_Mult(dr2dp, term_lnt2)
dvdt = Cplx_Sum(dvdt, vc)
vc = Cplx_Mult(dt1dp, term_t1)
vc = Cplx_Mult(vc, r1)
dvdt = Cplx_Sum(dvdt, vc)
vc = Cplx_Mult(dt2dp, term_t2)
vc = Cplx_Mult(vc, r2)

ice_d2gdtdp# = Cplx_Sum(dvdt, vc).Re * tau / Tt

Exit Function

Errortrap:
ice_d2gdtdp# = ice_error_return
End Function

Function ice_d3gdt2dp#(ByVal T#, ByVal P#)
On Error GoTo Errortrap

Dim CPC As CplxType
Dim CP0 As CplxType
Dim dr2dp As CplxType
Dim Tc As CplxType
Dim t2 As CplxType

Dim y#, z#, dz#

y# = T# / Tt#  'T in K
Tc = Cplx_Num(y)

z# = P# / Pt#  'P in Pa
dz# = (P# - P0#) / Pt#

dr2dp = Cplx_Factor(ddz_rk(2, dz#, 1), 1# / psi)
t2 = ddz_tk(2, dz#, 0)

CPC = Cplx_Div(dr2dp, Cplx_Diff(Tc, t2))
CPC = Cplx_Diff(CPC, Cplx_Div(dr2dp, Cplx_Sum(Tc, t2)))
CP0 = Cplx_Div(dr2dp, t2)

ice_d3gdt2dp# = -(2# * CP0.Re + CPC.Re) * tau / Tt ^ 2

Exit Function

Errortrap:
ice_d3gdt2dp# = ice_error_return
End Function


Function ice_d2gdp2#(ByVal T#, ByVal P#)
On Error GoTo Errortrap

Dim dr1dp As CplxType
Dim d2r2dp2 As CplxType
Dim dt1dp As CplxType
Dim dt2dp As CplxType
Dim Tc As CplxType
Dim r2 As CplxType
Dim t2 As CplxType
Dim tk As CplxType
Dim vc As CplxType
Dim term_tlnt2 As CplxType

Dim y#, z#, dz#, d2gdz2#

y# = T# / Tt#  'T in K
Tc = Cplx_Num(y)

z# = P# / Pt#  'P in Pa
dz# = (P# - P0#) / Pt#

r2 = ddz_rk(2, dz#, 0)
t2 = ddz_tk(2, dz#, 0)

d2r2dp2 = Cplx_Factor(ddz_rk(2, dz#, 2), 1 / psi ^ 2)

tk = Cplx_Diff(t2, Tc)
vc = Cplx_Mult(tk, Cplx_Log(tk))
tk = Cplx_Sum(Tc, t2)
vc = Cplx_Sum(vc, Cplx_Mult(tk, Cplx_Log(tk)))
tk = Cplx_Factor(Cplx_Mult(t2, Cplx_Log(t2)), 2)
vc = Cplx_Diff(vc, tk)

term_tlnt2 = Cplx_Diff(vc, Cplx_Div(Cplx_Num(y ^ 2), t2))
vc = Cplx_Mult(d2r2dp2, term_tlnt2)
d2gdz2# = ddz_poly(gcoeff(), dz, 2) / (psi ^ 2 * tau) + vc.Re

ice_d2gdp2# = tau * d2gdz2#

Exit Function

Errortrap:
ice_d2gdp2# = ice_error_return
End Function

Function ice_d3gdtdp2#(ByVal T#, ByVal P#)
On Error GoTo Errortrap

Dim d2r2dp2 As CplxType
Dim Tc As CplxType
Dim r2 As CplxType
Dim t2 As CplxType
Dim term_lnt2 As CplxType
Dim vc As CplxType

Dim y#, z#, dz#

y# = T# / Tt#  'T in K
Tc = Cplx_Num(y)

z# = P# / Pt#  'P in Pa
dz# = (P# - P0#) / Pt#

r2 = ddz_rk(2, dz#, 0)
t2 = ddz_tk(2, dz#, 0)

d2r2dp2 = Cplx_Factor(ddz_rk(2, dz#, 2), 1 / psi ^ 2)

term_lnt2 = Cplx_Diff(Cplx_Log(Cplx_Sum(t2, Tc)), Cplx_Log(Cplx_Diff(t2, Tc)))
term_lnt2 = Cplx_Diff(term_lnt2, Cplx_Factor(Cplx_Div(Tc, t2), 2))
vc = Cplx_Mult(d2r2dp2, term_lnt2)

ice_d3gdtdp2# = vc.Re * tau / Tt

Exit Function

Errortrap:
ice_d3gdtdp2# = ice_error_return
End Function


Function ice_free_enthalpy#(ByVal T#, ByVal P#)
'g(T,P) free enthalpy (Gibbs energy, chemical potential) in J/kg,
'T abs. temperature in K,
'P abs. pressure in Pa
ice_free_enthalpy = ice_g_FW04(0, 0, T, P)
End Function

Function ice_entropy#(ByVal T#, ByVal P#)
's(T,P) = -dg/dT entropy in J/(kg*K),
'T abs. temperature in K,
'P abs. pressure in Pa
ice_entropy = -ice_g_FW04(1, 0, T, P)
End Function

Function ice_enthalpy#(ByVal T#, ByVal P#)
'h(T,P) = g - T*(dg/dT) enthalpy in J/kg,
'T abs. temperature in K,
'P abs. pressure in Pa
ice_enthalpy = ice_g_FW04(0, 0, T, P) - T * ice_g_FW04(1, 0, T, P)
End Function


Function ice_specific_volume#(ByVal T#, ByVal P#)
'v(T,P) = dg/dP specific_volume in m^3/kg,
'T abs. temperature in K,
'P abs. pressure in Pa
ice_specific_volume = ice_g_FW04(0, 1, T, P)
End Function




Function ice_density#(ByVal T#, ByVal P#)
'rho(T,P) = 1/(dg/dP) density in kg/m^3,
'T abs. temperature in K,
'P abs. pressure in Pa
ice_density = 1# / ice_g_FW04(0, 1, T, P)
End Function
Function ice_free_energy#(ByVal T#, ByVal P#)
'f(T,P) = g - P*(dg/dP) free energy (Helmholtz energy) in J/kg,
'T abs. temperature in K,
'P abs. pressure in Pa
ice_free_energy# = ice_g_FW04(0, 0, T, P) - P * ice_g_FW04(0, 1, T, P)
End Function

Function ice_internal_energy#(ByVal T#, ByVal P#)
'u(T,P) = g - P*(dg/dP) - T*(dg/dT) internal energy in J/kg,
'T abs. temperature in K,
'P abs. pressure in Pa
ice_internal_energy = ice_g_FW04(0, 0, T, P) - P * ice_g_FW04(0, 1, T, P) - T * ice_g_FW04(1, 0, T, P)
End Function

Function ice_heat_capacity#(ByVal T#, ByVal P#)
'cp(T,P) =  - T*(d2g/dT2) isobaric heat capacity in J/(kg*K),
'T abs. temperature in K,
'P abs. pressure in Pa
ice_heat_capacity = -T * ice_g_FW04(2, 0, T, P)
End Function
Function ice_thermal_expansion#(ByVal T#, ByVal P#)
'alpha(T,P) = (d2g/dTdP)/(dg/dP) thermal expansion coefficient in 1/K,
'T abs. temperature in K,
'P abs. pressure in Pa
ice_thermal_expansion = ice_g_FW04(1, 1, T, P) / ice_g_FW04(0, 1, T, P)
End Function
Sub InitG_FW04()

'Check values of the Gibbs-Potential of Ice, Feistel & Wagner , J. Mar. Res. 2005
'Version of 31.01.2004 18:18:46

'Table of Coefficients
'g0(0) = -632578.704355102
'g0(1) = 0.655029997804786
'g0(2) = -1.89952376891314E-08
'g0(3) = 3.40692612753936E-15
'g0(4) = -5.78593658679522E-22
gcoeff(0) = -632578.704355102
gcoeff(1) = 0.655029997804786
gcoeff(2) = -1.89952376891314E-08
gcoeff(3) = 3.40692612753936E-15
gcoeff(4) = -5.78593658679522E-22

'residual entropy consistent with IAPWS-95 definition:
's0 = -3333.18160308627
scoeff(0) = -3333.18160308627

'alternative definition:
'absolute residual entropy (Pauling 1935, Nagle 1966):
's0 = 189.13
'scoeff(0) = 189.13

't1 = 3.71539090346389E-02 + i * 5.10464771184122E-02
'r1 = 45.951447199735 + i * 65.223705014775
tcoeff(1, 0) = Cplx_Num(3.71539090346389E-02, 5.10464771184122E-02)
rcoeff(1, 0) = Cplx_Num(45.951447199735, 65.223705014775)
rcoeff(1, 1) = Cplx_Num(0, 0)
rcoeff(1, 2) = Cplx_Num(0, 0)

't2 = 0.345095829562823 + i * 0.343315892017841
'r2(0) = -75.8695106343435 + i * -80.9878506462645
'r2(1) = -5.75529765634353E-05 + i * 5.09059011946526E-05
'r2(2) = 2.39617513518116E-11 + i * -2.73297877749166E-11
tcoeff(2, 0) = Cplx_Num(0.345095829562823, 0.343315892017841)
rcoeff(2, 0) = Cplx_Num(-75.8695106343435, -80.9878506462645)
rcoeff(2, 1) = Cplx_Num(-5.75529765634353E-05, 5.09059011946526E-05)
rcoeff(2, 2) = Cplx_Num(2.39617513518116E-11, -2.73297877749166E-11)

End Sub



